/**
  ******************************************************************************
  * @file    py32f032_hal_tim_ex.h
  * @author  MCU Application Team
  * @brief   Header file of TIM HAL Extended module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32F032_HAL_TIM_EX_H
#define __PY32F032_HAL_TIM_EX_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f032_hal_def.h"

/** @addtogroup PY32F032_HAL_Driver
  * @{
  */

/** @addtogroup TIMEx
  * @{
  */

/* Exported types ------------------------------------------------------------*/
/** @defgroup TIMEx_Exported_Types TIM Extended Exported Types
  * @{
  */

/**
  * @brief  TIM Hall sensor Configuration Structure definition
  */

typedef struct
{
  uint32_t IC1Polarity;         /*!< Specifies the active edge of the input signal.
                                     This parameter can be a value of @ref TIM_Input_Capture_Polarity */

  uint32_t IC1Prescaler;        /*!< Specifies the Input Capture Prescaler.
                                     This parameter can be a value of @ref TIM_Input_Capture_Prescaler */

  uint32_t IC1Filter;           /*!< Specifies the input capture filter.
                                     This parameter can be a number between Min_Data = 0x0 and Max_Data = 0xF */

  uint32_t Commutation_Delay;   /*!< Specifies the pulse value to be loaded into the Capture Compare Register. */
} TIM_HallSensor_InitTypeDef;

/**
  * @brief  TIM Break input configuration
  */
typedef struct
{
  uint32_t Source;         /*!< Specifies the source of the timer break input.
                                This parameter can be a value of @ref TIMEx_Break_Input_Source */
  uint32_t Enable;         /*!< Specifies whether or not the break input source is enabled.
                                This parameter can be a value of @ref TIMEx_Break_Input_Source_Enable */
  uint32_t Polarity;       /*!< Specifies the break input source polarity.
                                This parameter can be a value of @ref TIMEx_Break_Input_Source_Polarity */
} TIMEx_BreakInputConfigTypeDef;
/**
  * @}
  */
/* End of exported types -----------------------------------------------------*/

/* Exported constants --------------------------------------------------------*/
/** @defgroup TIMEx_Exported_Constants TIM Extended Exported Constants
  * @{
  */

/** @defgroup TIMEx_Remap TIM Extended Remapping
  * @{
  */
#define TIM_TIM1_ETR_GPIO        0x00000000U                                                                     /*!< ETR input is connected to GPIO */
#define TIM_TIM1_ETR_COMP1       (                                                            TIM_AF1_ETRSEL_0)  /*!< ETR input is connected to COMP1_OUT */
#define TIM_TIM1_ETR_COMP2       (                                        TIM_AF1_ETRSEL_1                    )  /*!< ETR input is connected to COMP2_OUT */
#define TIM_TIM1_ETR_ADC_AWD     (                    TIM_AF1_ETRSEL_2                      | TIM_AF1_ETRSEL_0)  /*!< ETR input is connected to ADC analog watchdogg */

#define TIM_TIM3_ETR_GPIO        0x00000000U                                                                    /*!< ETR input is connected to GPIO */
#define TIM_TIM3_ETR_COMP1       (                                                            TIM_AF1_ETRSEL_0) /*!< ETR input is connected to COMP1_OUT */
#define TIM_TIM3_ETR_COMP2       (                                        TIM_AF1_ETRSEL_1                    ) /*!< ETR input is connected to COMP2_OUT */
#define TIM_TIM3_ETR_ADC_AWD     (                    TIM_AF1_ETRSEL_2  | TIM_AF1_ETRSEL_1                    ) /*!< ETR input is connected to ADC analog watchdog */

/**
  * @}
  */

/** @defgroup TIMEx_Break_Input TIM Extended Break input
  * @{
  */
#define TIM_BREAKINPUT_BRK     0x00000001U                                      /*!< Timer break input  */
/**
  * @}
  */

/** @defgroup TIMEx_Break_Input_Source TIM Extended Break input source
  * @{
  */
#define TIM_BREAKINPUTSOURCE_BKIN     0x00000001U                               /*!< An external source (GPIO) is connected to the BKIN pin  */
#define TIM_BREAKINPUTSOURCE_COMP1    0x00000002U                               /*!< The COMP1 output is connected to the break input */
#define TIM_BREAKINPUTSOURCE_COMP2    0x00000004U                               /*!< The COMP2 output is connected to the break input */

/**
  * @}
  */

/** @defgroup TIMEx_Break_Input_Source_Enable TIM Extended Break input source enabling
  * @{
  */
#define TIM_BREAKINPUTSOURCE_DISABLE     0x00000000U                            /*!< Break input source is disabled */
#define TIM_BREAKINPUTSOURCE_ENABLE      0x00000001U                            /*!< Break input source is enabled */
/**
  * @}
  */

/** @defgroup TIMEx_Break_Input_Source_Polarity TIM Extended Break input polarity
  * @{
  */
#define TIM_BREAKINPUTSOURCE_POLARITY_LOW     0x00000001U                       /*!< Break input source is active low */
#define TIM_BREAKINPUTSOURCE_POLARITY_HIGH    0x00000000U                       /*!< Break input source is active_high */
/**
  * @}
  */

/** @defgroup TIMEx_Timer_Input_Selection TIM Extended Timer input selection
  * @{
  */
#define TIM_TIM1_TI1_GPIO     0x00000000U                                                    /*!< TIM1 input 1 is connected to GPIO */
#define TIM_TIM1_TI1_COMP1    (                                          TIM_TISEL_TI1SEL_0) /*!< TIM1 input 1 is connected to COMP1_OUT */
#define TIM_TIM1_TI1_COMP2    (                     TIM_TISEL_TI1SEL_1                     ) /*!< TIM1 input 1 is connected to COMP2_OUT */

#define TIM_TIM1_TI2_GPIO     0x00000000U                                                    /*!< TIM1 input 2 is connected to GPIO */
#define TIM_TIM1_TI2_COMP1    (                                          TIM_TISEL_TI2SEL_0) /*!< TIM1 input 2 is connected to COMP1_OUT */
#define TIM_TIM1_TI2_COMP2    (                     TIM_TISEL_TI2SEL_1                     ) /*!< TIM1 input 2 is connected to COMP2_OUT */

#define TIM_TIM1_TI3_GPIO     0x00000000U                                                    /*!< TIM1 input 3 is connected to GPIO */
#define TIM_TIM1_TI3_COMP1    (                                          TIM_TISEL_TI3SEL_0) /*!< TIM1 input 3 is connected to COMP1_OUT */
#define TIM_TIM1_TI3_COMP2    (                     TIM_TISEL_TI3SEL_1                     ) /*!< TIM1 input 3 is connected to COMP2_OUT */

#define TIM_TIM1_TI4_GPIO     0x00000000U                                                    /*!< TIM1 input 4 is connected to GPIO */
#define TIM_TIM1_TI4_COMP1    (                                          TIM_TISEL_TI4SEL_0) /*!< TIM1 input 4 is connected to COMP1_OUT */
#define TIM_TIM1_TI4_COMP2    (                     TIM_TISEL_TI4SEL_1                     ) /*!< TIM1 input 4 is connected to COMP2_OUT */

#define TIM_TIM3_TI1_GPIO     0x00000000U                                                    /*!< TIM3 input 1 is connected to GPIO */
#define TIM_TIM3_TI1_COMP1    (                                          TIM_TISEL_TI1SEL_0) /*!< TIM3 input 1 is connected to COMP1_OUT */
#define TIM_TIM3_TI1_COMP2    (                     TIM_TISEL_TI1SEL_1                     ) /*!< TIM3 input 1 is connected to COMP2_OUT */

#define TIM_TIM3_TI2_GPIO     0x00000000U                                                    /*!< TIM3 input 2 is connected to GPIO */
#define TIM_TIM3_TI2_COMP1    (                                          TIM_TISEL_TI2SEL_0) /*!< TIM3 input 2 is connected to COMP1_OUT */
#define TIM_TIM3_TI2_COMP2    (                     TIM_TISEL_TI2SEL_1                     ) /*!< TIM3 input 2 is connected to COMP2_OUT */

#define TIM_TIM3_TI3_GPIO     0x00000000U                                                    /*!< TIM3 input 3 is connected to GPIO */
#define TIM_TIM3_TI3_COMP1    (                                          TIM_TISEL_TI3SEL_0) /*!< TIM3 input 3 is connected to COMP1_OUT */
#define TIM_TIM3_TI3_COMP2    (                     TIM_TISEL_TI3SEL_1                     ) /*!< TIM3 input 3 is connected to COMP2_OUT */

#define TIM_TIM3_TI4_GPIO     0x00000000U                                                    /*!< TIM3 input 4 is connected to GPIO */
#define TIM_TIM3_TI4_COMP1    (                                          TIM_TISEL_TI4SEL_0) /*!< TIM3 input 4 is connected to COMP1_OUT */
#define TIM_TIM3_TI4_COMP2    (                     TIM_TISEL_TI4SEL_1                     ) /*!< TIM3 input 4 is connected to COMP2_OUT */

#define TIM_TIM14_TI1_GPIO    0x00000000U                                                    /*!< TIM15 input 1 is connected to GPIO */
#define TIM_TIM14_TI1_RTCCLK  (                                          TIM_TISEL_TI1SEL_0) /*!< TIM15 input 1 is connected to LSE_CSS */
#define TIM_TIM14_TI1_HSE_32  (                     TIM_TISEL_TI1SEL_1                     ) /*!< TIM15 input 1 is connected to COMP1_OUT */
#define TIM_TIM14_TI1_MCO     (                     TIM_TISEL_TI1SEL_1 | TIM_TISEL_TI1SEL_0) /*!< TIM15 input 1 is connected to COMP2_OUT */


#define TIM_TIM16_TI1_GPIO    0x00000000U                                                    /*!< TIM16 input 1 is connected to GPIO */
#define TIM_TIM16_TI1_MCO     (                     TIM_TISEL_TI1SEL_1                     ) /*!< TIM16 input 1 is connected to MCO */
#define TIM_TIM16_TI1_HSE_32  (                     TIM_TISEL_TI1SEL_1 | TIM_TISEL_TI1SEL_0) /*!< TIM16 input 1 is connected to HSE/32 */
#define TIM_TIM16_TI1_RTC_WUK (TIM_TISEL_TI1SEL_2                                          ) /*!< TIM16 input 1 is connected to RTC */
#define TIM_TIM16_TI1_LSE_CSS (TIM_TISEL_TI1SEL_2                      | TIM_TISEL_TI1SEL_0) /*!< TIM16 input 1 is connected to LSE_CSS */
#define TIM_TIM16_TI1_LSI     (TIM_TISEL_TI1SEL_2 | TIM_TISEL_TI1SEL_1                     ) /*!< TIM16 input 1 is connected to LSI */


#define TIM_TIM17_TI1_GPIO    0x00000000U                                                    /*!< TIM17 input 1 is connected to GPIO */
#define TIM_TIM17_TI1_MCO     (                     TIM_TISEL_TI1SEL_1                     ) /*!< TIM17 input 1 is connected to MCO */
#define TIM_TIM17_TI1_HSE_32  (                     TIM_TISEL_TI1SEL_1 | TIM_TISEL_TI1SEL_0) /*!< TIM17 input 1 is connected to HSE/32 */
#define TIM_TIM17_TI1_RTC_WUK (TIM_TISEL_TI1SEL_2                                          ) /*!< TIM17 input 1 is connected to RTC */
#define TIM_TIM17_TI1_LSE_CSS (TIM_TISEL_TI1SEL_2                      | TIM_TISEL_TI1SEL_0) /*!< TIM17 input 1 is connected to LSE_CSS */
#define TIM_TIM17_TI1_LSI     (TIM_TISEL_TI1SEL_2 | TIM_TISEL_TI1SEL_1                     ) /*!< TIM17 input 1 is connected to LSI */

/**
  * @}
  */


/**
  * @}
  */
/* End of exported constants -------------------------------------------------*/

/* Exported macro ------------------------------------------------------------*/
/** @defgroup TIMEx_Exported_Macros TIM Extended Exported Macros
  * @{
  */

/**
  * @}
  */
/* End of exported macro -----------------------------------------------------*/

/* Private macro -------------------------------------------------------------*/
/** @defgroup TIMEx_Private_Macros TIM Extended Private Macros
  * @{
  */
#define IS_TIM_REMAP(__REMAP__) ((((__REMAP__) & 0xFFFC3FFFU) == 0x00000000U))

#define IS_TIM_BREAKINPUT(__BREAKINPUT__)  ((__BREAKINPUT__) == TIM_BREAKINPUT_BRK) 

#define IS_TIM_BREAKINPUTSOURCE(__SOURCE__)  (((__SOURCE__) == TIM_BREAKINPUTSOURCE_BKIN)  || \
                                              ((__SOURCE__) == TIM_BREAKINPUTSOURCE_COMP1) || \
                                              ((__SOURCE__) == TIM_BREAKINPUTSOURCE_COMP2))

#define IS_TIM_BREAKINPUTSOURCE_STATE(__STATE__)  (((__STATE__) == TIM_BREAKINPUTSOURCE_DISABLE)  || \
                                                   ((__STATE__) == TIM_BREAKINPUTSOURCE_ENABLE))

#define IS_TIM_BREAKINPUTSOURCE_POLARITY(__POLARITY__)  (((__POLARITY__) == TIM_BREAKINPUTSOURCE_POLARITY_LOW)  || \
                                                         ((__POLARITY__) == TIM_BREAKINPUTSOURCE_POLARITY_HIGH))

#define IS_TIM_TISEL(__TISEL__) ((((__TISEL__) & 0xF0F0F0F0U) == 0x00000000U))

#define IS_TIM_TISEL_TIX_INSTANCE(INSTANCE, CHANNEL) \
  (IS_TIM_CCX_INSTANCE(INSTANCE, CHANNEL) && ((CHANNEL) <= TIM_CHANNEL_4))

/**
  * @}
  */
/* End of private macro ------------------------------------------------------*/

/* Exported functions --------------------------------------------------------*/
/** @addtogroup TIMEx_Exported_Functions TIM Extended Exported Functions
  * @{
  */

/** @addtogroup TIMEx_Exported_Functions_Group1 Extended Timer Hall Sensor functions
  *  @brief    Timer Hall Sensor functions
  * @{
  */
/*  Timer Hall Sensor functions  **********************************************/
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Init(TIM_HandleTypeDef *htim, TIM_HallSensor_InitTypeDef *sConfig);
HAL_StatusTypeDef HAL_TIMEx_HallSensor_DeInit(TIM_HandleTypeDef *htim);

void HAL_TIMEx_HallSensor_MspInit(TIM_HandleTypeDef *htim);
void HAL_TIMEx_HallSensor_MspDeInit(TIM_HandleTypeDef *htim);

/* Blocking mode: Polling */
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Start(TIM_HandleTypeDef *htim);
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Stop(TIM_HandleTypeDef *htim);
/* Non-Blocking mode: Interrupt */
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Start_IT(TIM_HandleTypeDef *htim);
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Stop_IT(TIM_HandleTypeDef *htim);
/* Non-Blocking mode: DMA */
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Start_DMA(TIM_HandleTypeDef *htim, uint32_t *pData, uint16_t Length);
HAL_StatusTypeDef HAL_TIMEx_HallSensor_Stop_DMA(TIM_HandleTypeDef *htim);
/**
  * @}
  */

/** @addtogroup TIMEx_Exported_Functions_Group2 Extended Timer Complementary Output Compare functions
  *  @brief   Timer Complementary Output Compare functions
  * @{
  */
/*  Timer Complementary Output Compare functions  *****************************/
/* Blocking mode: Polling */
HAL_StatusTypeDef HAL_TIMEx_OCN_Start(TIM_HandleTypeDef *htim, uint32_t Channel);
HAL_StatusTypeDef HAL_TIMEx_OCN_Stop(TIM_HandleTypeDef *htim, uint32_t Channel);

/* Non-Blocking mode: Interrupt */
HAL_StatusTypeDef HAL_TIMEx_OCN_Start_IT(TIM_HandleTypeDef *htim, uint32_t Channel);
HAL_StatusTypeDef HAL_TIMEx_OCN_Stop_IT(TIM_HandleTypeDef *htim, uint32_t Channel);

/* Non-Blocking mode: DMA */
HAL_StatusTypeDef HAL_TIMEx_OCN_Start_DMA(TIM_HandleTypeDef *htim, uint32_t Channel, uint32_t *pData, uint16_t Length);
HAL_StatusTypeDef HAL_TIMEx_OCN_Stop_DMA(TIM_HandleTypeDef *htim, uint32_t Channel);
/**
  * @}
  */

/** @addtogroup TIMEx_Exported_Functions_Group3 Extended Timer Complementary PWM functions
  *  @brief    Timer Complementary PWM functions
  * @{
  */
/*  Timer Complementary PWM functions  ****************************************/
/* Blocking mode: Polling */
HAL_StatusTypeDef HAL_TIMEx_PWMN_Start(TIM_HandleTypeDef *htim, uint32_t Channel);
HAL_StatusTypeDef HAL_TIMEx_PWMN_Stop(TIM_HandleTypeDef *htim, uint32_t Channel);

/* Non-Blocking mode: Interrupt */
HAL_StatusTypeDef HAL_TIMEx_PWMN_Start_IT(TIM_HandleTypeDef *htim, uint32_t Channel);
HAL_StatusTypeDef HAL_TIMEx_PWMN_Stop_IT(TIM_HandleTypeDef *htim, uint32_t Channel);
/* Non-Blocking mode: DMA */
HAL_StatusTypeDef HAL_TIMEx_PWMN_Start_DMA(TIM_HandleTypeDef *htim, uint32_t Channel, uint32_t *pData, uint16_t Length);
HAL_StatusTypeDef HAL_TIMEx_PWMN_Stop_DMA(TIM_HandleTypeDef *htim, uint32_t Channel);
/**
  * @}
  */

/** @addtogroup TIMEx_Exported_Functions_Group4 Extended Timer Complementary One Pulse functions
  *  @brief    Timer Complementary One Pulse functions
  * @{
  */
/*  Timer Complementary One Pulse functions  **********************************/
/* Blocking mode: Polling */
HAL_StatusTypeDef HAL_TIMEx_OnePulseN_Start(TIM_HandleTypeDef *htim, uint32_t OutputChannel);
HAL_StatusTypeDef HAL_TIMEx_OnePulseN_Stop(TIM_HandleTypeDef *htim, uint32_t OutputChannel);

/* Non-Blocking mode: Interrupt */
HAL_StatusTypeDef HAL_TIMEx_OnePulseN_Start_IT(TIM_HandleTypeDef *htim, uint32_t OutputChannel);
HAL_StatusTypeDef HAL_TIMEx_OnePulseN_Stop_IT(TIM_HandleTypeDef *htim, uint32_t OutputChannel);
/**
  * @}
  */

/** @addtogroup TIMEx_Exported_Functions_Group5 Extended Peripheral Control functions
  *  @brief    Peripheral Control functions
  * @{
  */
/* Extended Control functions  ************************************************/
HAL_StatusTypeDef HAL_TIMEx_ConfigCommutEvent(TIM_HandleTypeDef *htim, uint32_t  InputTrigger,
    uint32_t  CommutationSource);
HAL_StatusTypeDef HAL_TIMEx_ConfigCommutEvent_IT(TIM_HandleTypeDef *htim, uint32_t  InputTrigger,
    uint32_t  CommutationSource);
HAL_StatusTypeDef HAL_TIMEx_ConfigCommutEvent_DMA(TIM_HandleTypeDef *htim, uint32_t  InputTrigger,
    uint32_t  CommutationSource);
HAL_StatusTypeDef HAL_TIMEx_MasterConfigSynchronization(TIM_HandleTypeDef *htim,
    TIM_MasterConfigTypeDef *sMasterConfig);
HAL_StatusTypeDef HAL_TIMEx_ConfigBreakDeadTime(TIM_HandleTypeDef *htim,
    TIM_BreakDeadTimeConfigTypeDef *sBreakDeadTimeConfig);
HAL_StatusTypeDef HAL_TIMEx_ConfigBreakInput(TIM_HandleTypeDef *htim, uint32_t BreakInput,
                                             const TIMEx_BreakInputConfigTypeDef *sBreakInputConfig);
HAL_StatusTypeDef HAL_TIMEx_RemapConfig(TIM_HandleTypeDef *htim, uint32_t Remap);
HAL_StatusTypeDef  HAL_TIMEx_TISelection(TIM_HandleTypeDef *htim, uint32_t TISelection, uint32_t Channel);
HAL_StatusTypeDef HAL_TIMEx_ConfigDeadTime(TIM_HandleTypeDef *htim, uint32_t Deadtime);
/**
  * @}
  */

/** @addtogroup TIMEx_Exported_Functions_Group6 Extended Callbacks functions
  * @brief    Extended Callbacks functions
  * @{
  */
/* Extended Callback **********************************************************/
void HAL_TIMEx_CommutCallback(TIM_HandleTypeDef *htim);
void HAL_TIMEx_CommutHalfCpltCallback(TIM_HandleTypeDef *htim);
void HAL_TIMEx_BreakCallback(TIM_HandleTypeDef *htim);
/**
  * @}
  */

/** @addtogroup TIMEx_Exported_Functions_Group7 Extended Peripheral State functions
  * @brief    Extended Peripheral State functions
  * @{
  */
/* Extended Peripheral State functions  ***************************************/
HAL_TIM_StateTypeDef HAL_TIMEx_HallSensor_GetState(TIM_HandleTypeDef *htim);
/**
  * @}
  */

/**
  * @}
  */
/* End of exported functions -------------------------------------------------*/

/* Private functions----------------------------------------------------------*/
/** @addtogroup TIMEx_Private_Functions TIM Extended Private Functions
  * @{
  */
#if (defined(DMA) || defined(DMA1))
void TIMEx_DMACommutationCplt(DMA_HandleTypeDef *hdma);
void TIMEx_DMACommutationHalfCplt(DMA_HandleTypeDef *hdma);
#endif
/**
  * @}
  */
/* End of private functions --------------------------------------------------*/

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif


#endif /* __PY32F032_HAL_TIM_EX_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
